<?php

if (!defined('ABSPATH')) exit;

class ACE_WebP_Converter
{
    private string $method = '';

    public function __construct()
    {
        if (function_exists('imagecreatefromjpeg') && function_exists('imagewebp')) {
            $this->method = 'gd';
        } elseif (class_exists('Imagick')) {
            try {
                $formats = \Imagick::queryFormats('WEBP');
                if (!empty($formats)) {
                    $this->method = 'imagick';
                }
            } catch (\Exception $e) {
                // Imagick not properly configured.
            }
        }
    }

    public function is_supported(): bool
    {
        return $this->method !== '';
    }

    public function get_method(): string
    {
        return strtoupper($this->method) ?: 'None';
    }

    /**
     * Convert a single image file to WebP.
     *
     * @return string|false WebP file path on success, false on skip/failure.
     */
    public function convert_file(string $file_path, int $quality = 82): string|false
    {
        if (!file_exists($file_path)) return false;

        $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg', 'jpeg', 'png'], true)) return false;

        // Skip files smaller than 10 KB.
        if (filesize($file_path) < 10240) return false;

        $webp_path = $file_path . '.webp';

        // Already converted.
        if (file_exists($webp_path)) return $webp_path;

        $result = match ($this->method) {
            'gd'      => $this->convert_gd($file_path, $webp_path, $ext, $quality),
            'imagick' => $this->convert_imagick($file_path, $webp_path, $ext, $quality),
            default   => false,
        };

        if ($result === false || !file_exists($webp_path)) return false;

        // Discard if WebP is not smaller than the original.
        if (filesize($webp_path) >= filesize($file_path)) {
            unlink($webp_path);
            return false;
        }

        return $webp_path;
    }

    /**
     * Convert all sizes of a media attachment.
     */
    public function convert_attachment(int $attachment_id, array $metadata): void
    {
        $file = get_attached_file($attachment_id);
        if (!$file || !file_exists($file)) return;

        $quality = (int) get_option('ace_webp_quality', 82);

        // Convert original.
        $this->convert_file($file, $quality);

        // Convert each registered thumbnail size.
        if (!empty($metadata['sizes'])) {
            $dir = dirname($file);
            foreach ($metadata['sizes'] as $size) {
                $this->convert_file($dir . '/' . $size['file'], $quality);
            }
        }
    }

    /* ------------------------------------------------------------------
     * GD conversion
     * ----------------------------------------------------------------*/
    private function convert_gd(string $source, string $dest, string $ext, int $quality): bool
    {
        $image = match ($ext) {
            'jpg', 'jpeg' => @imagecreatefromjpeg($source),
            'png'         => @imagecreatefrompng($source),
            default       => false,
        };

        if (!$image) return false;

        if ($ext === 'png') {
            imagepalettetotruecolor($image);
            imagealphablending($image, true);
            imagesavealpha($image, true);
        }

        $ok = imagewebp($image, $dest, $quality);
        imagedestroy($image);

        return $ok;
    }

    /* ------------------------------------------------------------------
     * Imagick conversion
     * ----------------------------------------------------------------*/
    private function convert_imagick(string $source, string $dest, string $ext, int $quality): bool
    {
        try {
            $img = new \Imagick($source);

            if ($ext === 'png') {
                $img->setImageAlphaChannel(\Imagick::ALPHACHANNEL_ACTIVATE);
                $img->setBackgroundColor(new \ImagickPixel('transparent'));
            }

            $img->setImageFormat('webp');
            $img->setImageCompressionQuality($quality);
            $ok = $img->writeImage($dest);
            $img->clear();
            $img->destroy();

            return $ok;
        } catch (\Exception $e) {
            return false;
        }
    }
}
